/* sha1.c
   Module for SHA-1 calculation in accordance with RFC 3174
   Copyright (C) 2003 Linus Walleij

   This file is part of the shftool package.

   Shftool is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   You should have received a copy of the GNU General Public License
   along with Shftool; see the file COPYING.  If not, write to
   the Free Software Foundation, 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA. 

*/

#include "common.h"

static u_int8_t buffer[64];
static u_int8_t bufpoint;
static u_int32_t h0, h1, h2, h3, h4;

static void zerobuffer()
{
  int i;

  for (i = 0; i < 64; i++) {
    buffer[i] = 0;
  }
  bufpoint = 0;
}

void initialize_sha1()
{
  h0 = 0x67452301;
  h1 = 0xEFCDAB89;
  h2 = 0x98BADCFE;
  h3 = 0x10325476;
  h4 = 0xC3D2E1F0;
  zerobuffer();
}

#define Rotate32(bits,word) (((word) << (bits)) | ((word) >> (32-(bits))))
static void calculate_sha1_for_buffer() {
  // Calculate SHA-1 for the buffer.
  int i;
  u_int32_t chunks[80];
  u_int8_t *ret;
  u_int32_t a, b, c, d, e, f, k, t;

  // OutString("\nBuffer: ");
  // dumpHexArray(buffer, 64);
  // OutString("\n");

  // Initialize variables...
  a = h0;
  b = h1;
  c = h2;
  d = h3;
  e = h4;

  // Chop bytes into 32-bit chunks
  for (i = 0; i < 16; i++) {
    int bp = i << 2;
    chunks[i] = (buffer[bp] << 24) | (buffer[bp+1] << 16) | (buffer[bp+2] << 8) | buffer[bp+3];
  }
  // Extend them to 80 32-bit chunks
  for (i = 16; i < 80; i++) {
    chunks[i] = Rotate32(1,(chunks[i-3] ^ chunks[i-8] ^ chunks[i-14] ^ chunks[i-16]));
  }
  // Do the stuff
  for (i = 0; i < 80; i++){
    
    if (i < 20) {
      //f = ((b & c) | ((~b) & d));
      f = (d ^ (b & (c ^ d)));
      k = 0x5A827999;
    } else if (i < 40) {
      f = b ^ c ^ d;
      k = 0x6ED9EBA1;
    } else if (i < 60) {
      //f = ((b & c) | (b & d) | (c & d));
      f = (b & c) | (d & (b | c));
      k = 0x8F1BBCDC;
    } else {
      f = b ^ c ^ d;
      k = 0xCA62C1D6;
    }
    t = Rotate32(5,a) + f + e + k + chunks[i];
    e = d;
    d = c;
    c = Rotate32(30,b);
    b = a;
    a = t;
  }

  // Add to total digest
  h0 += a;
  h1 += b;
  h2 += c;
  h3 += d;
  h4 += e;
}


void add_byte_to_sha1(u_int8_t c) 
{
  buffer[bufpoint] = c;
  bufpoint ++;
  // Time to do the stuff?
  if (bufpoint == 64) {
    calculate_sha1_for_buffer();
    // Then reset the buffer.
    zerobuffer();
  }
}

u_int8_t *retrieve_sha1(u_int64_t msglen)
{
  u_int8_t *ret = malloc(20);

  // Add 1 on the end
  buffer[bufpoint] = 0x80;
  bufpoint ++;
  if (bufpoint > 56) {
    calculate_sha1_for_buffer();
    zerobuffer();
  }
  buffer[56] = (msglen >> 56) & 0xff;
  buffer[57] = (msglen >> 48) & 0xff;
  buffer[58] = (msglen >> 40) & 0xff;
  buffer[59] = (msglen >> 32) & 0xff;
  buffer[60] = (msglen >> 24) & 0xff;
  buffer[61] = (msglen >> 16) & 0xff;
  buffer[62] = (msglen >> 8) & 0xff;
  buffer[63] = msglen & 0xff;


  // Perform last checksumming.
  calculate_sha1_for_buffer();

  ret[0] = (h0 >> 24) & 0xff;
  ret[1] = (h0 >> 16) & 0xff;
  ret[2] = (h0 >> 8) & 0xff;
  ret[3] = h0 & 0xff;
  ret[4] = (h1 >> 24) & 0xff;
  ret[5] = (h1 >> 16) & 0xff;
  ret[6] = (h1 >> 8) & 0xff;
  ret[7] = h1 & 0xff;
  ret[8] = (h2 >> 24) & 0xff;
  ret[9] = (h2 >> 16) & 0xff;
  ret[10] = (h2 >> 8) & 0xff;
  ret[11] = h2 & 0xff;
  ret[12] = (h3 >> 24) & 0xff;
  ret[13] = (h3 >> 16) & 0xff;
  ret[14] = (h3 >> 8) & 0xff;
  ret[15] = h3 & 0xff;
  ret[16] = (h4 >> 24) & 0xff;
  ret[17] = (h4 >> 16) & 0xff;
  ret[18] = (h4 >> 8) & 0xff;
  ret[19] = h4 & 0xff;
  return ret;
}
