/* parseshf.c
   Program for parsing and converting shf.dtd compliant 
   XML files to raw memory blocks, to be processed by a writer.
   Requires Expat (commonly packaged as xmltok-dev)
   Copyright (C) 2003 Linus Walleij


   This file is part of the shftool package.

   Shftool is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   You should have received a copy of the GNU General Public License
   along with Shftool; see the file COPYING.  If not, write to
   the Free Software Foundation, 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA. 

*/

#include "common.h"

// Has the dump begun?
static bool DumpStarted = false;
static bool DataThroughput = false;

// Specifics about this dump
static shf_block_t *currentblock = NULL;
static u_int64_t blockcount;
static u_int64_t readblocks;

// Data word scanning
static int currentnybble;
static u_int8_t currenthinybble;
static u_int8_t currentlonybble;

static u_int8_t *checksumFromStr(const char *str) {
  int i, j, k;
  u_int8_t hinybble;
  u_int8_t lonybble;
  u_int8_t *checksum;

  checksum = (u_int8_t *) malloc(20);
  j = 0;
  k = 0;
  for (i = 0; i < strlen(str); i++) {
    // Remove illegal characters
    if (filterHex(str[i]) != '\0') {
      u_int8_t b = hexToNybble(str[i]);
      if (j == 0) {
	hinybble = b << 4;
	j ++;
      } else {
	lonybble = b;
	checksum[k] = hinybble | lonybble;
	if (k < 20) {
	  k++;
	} else {
	  OutError("Invalid block checksum size. Aborting.");
	  exit(1);
	}
	j = 0;
      }
    }
  }
  if (k != 20) {
    OutError("Invalid block checksum size. Aborting.");
    exit(1);
  }
  return checksum;
}

// Start handler for XML elements
static void startElementHandler(void *data, const XML_Char *el, const XML_Char **attr) 
{
  int i;

  if (!strcmp(el,"dump")) {
    char *name = NULL;

    DumpStarted = true;
    readblocks = 0;
    for (i = 0; attr[i]; i += 2) {
      if (!strcmp(attr[i], "name"))
	name = strdup(attr[i+1]);
      else if (!strcmp(attr[i], "blocks")) {
	blockcount = hexStrToULint(attr[i+1]);
      }
      else {
	OutError(" Invalid attribute for \"dump\" element: %s, value: %s\n", attr[i], attr[i+1]);
      }
    }
    write_header(name, blockcount);
    free(name);
  }
  else if (!strcmp(el,"block")) {
    shf_block_t *block;
    char *blockname;
    u_int64_t blockaddress;
    u_int64_t blockwordlength;
    u_int64_t blocklength;
    u_int8_t *blockchecksum;
    u_int8_t *data;
 
    readblocks ++;
    initialize_sha1();

    for (i = 0; attr[i]; i += 2) {
      if (!strcmp(attr[i], "name"))
	blockname = strdup(attr[i+1]);
      else if (!strcmp(attr[i], "address")) {
	blockaddress = hexStrToULint(attr[i+1]);
      }
      else if (!strcmp(attr[i], "word_size")) {
	blockwordlength = hexStrToULint(attr[i+1]);
	// currentword = malloc(wordlen);
      }
      else if (!strcmp(attr[i], "length")) {
	blocklength = hexStrToULint(attr[i+1]);
      }
      else if (!strcmp(attr[i], "checksum")) {
	blockchecksum = checksumFromStr(attr[i+1]);
      }
      else {
	OutError("Invalid attribute for \"block\" element: %s, value: %s\n", attr[i], attr[i+1]);
      }
    }
    // Create block data structure
    block = (shf_block_t *) malloc(sizeof(shf_block_t));
    block->number = readblocks;
    block->name = blockname;
    block->address = blockaddress;
    block->word_length = blockwordlength;
    block->length = blocklength;
    block->checksum = blockchecksum;
    block->datap = 0;
    block->data = (u_int8_t *) malloc(blockwordlength * blocklength);
    // Make this global
    currentblock = block;
  }
  else if (!strcmp(el,"data")) {
    // Start handling data
    DataThroughput = true;
    currentnybble = 0;
    currentblock->datap = 0;
    for (i = 0; attr[i]; i += 2) {
      OutError("Invalid attribute for \"data\" element: %s, value: %s\n", attr[i], attr[i+1]);
    }
  }
  else {
    OutError("Invalid element: %s", el);
    for (i = 0; attr[i]; i += 2) {
      OutError("Invalid attribute: %, value: %s\n", attr[i], attr[i+1]);
    }
  }
}


// End handler for XML elements
static void endElementHandler(void *data, const XML_Char *el)
{
  if (!strcmp(el,"dump")) {
    DumpStarted = false;
    if (readblocks != blockcount) {
      OutError("Incorrect number of blocks! Expected %llx got %llx\n", blockcount, readblocks);
    }
    write_footer();
  } else if (!strcmp(el,"block")) {
    // Block ended...
    u_int8_t *calcsum;
    int i;

    if (currentblock->datap != (currentblock->word_length * currentblock->length)) {
      OutError("Incorrect number of words! Expected %llx words in %llx bytes, got %llx\n", 
		currentblock->length, 
		(currentblock->length * currentblock->word_length), 
		currentblock->datap);
    }
    calcsum = retrieve_sha1(currentblock->length * currentblock->word_length * 8);
    for (i = 0; i < 20; i++) {
      if (calcsum[i] != currentblock->checksum[i]) {
	OutError("Erroneous checksum on block %llx!\n", currentblock->number);
	OutError("Tagged checksum:     ");
	dumpHexArray(currentblock->checksum, 20, true);
	OutError("\n");
	OutError("Calculated checksum: ");
	dumpHexArray(calcsum, 20, true);
	OutError("\n");
	destroy_block_t(currentblock);
	exit(1);
      }
    }
    free(calcsum);
    write_block(currentblock);
    destroy_block_t(currentblock);
  } else if (!strcmp(el,"data")) {
    DataThroughput = false;
  } else {
    OutError("End of invalid element: %s", el);
  }
}


// The raw text falls through to here
static void textHandler(void *data, const XML_Char *text, int length)
{
  if (!DumpStarted) {
    // Do nothing of the Dump hasn't started
  } else if (DataThroughput) {
    // Wow dump data!
    int i;
    char inchar;

    for (i=0;i<length;i++) {
      inchar = (char) text[i];
      // Silently ignore anything else than [0-9][a-f][A-F]
      if (filterHex(inchar) != '\0') {
	if (currentnybble == 0) {
	  currenthinybble = hexToNybble(inchar) << 4;
	  currentnybble ++;
	} else {
	  u_int8_t currentbyte;

	  currentlonybble = hexToNybble(inchar);
	  currentbyte = currenthinybble | currentlonybble;
	  currentblock->data[currentblock->datap] = currentbyte;
	  currentblock->datap++;
	  // Error message and exit on buffer overflow
	  if (currentblock->datap > (currentblock->length * currentblock->word_length)) {
	    OutError("Incorrect number of words - too many! Expected %llx\n", currentblock->length);
	    destroy_block_t(currentblock);
	    exit(1);
	  }
	  add_byte_to_sha1(currentbyte);
	  currentnybble = 0;
	}
      }
    }
  }
}

// All raw entities are handled here
static void defaultHandler(void *userData, const XML_Char *s, int length)
{
  register int i;
  static char temp_string_1[15]; /* How long can an escape code be? */
  char *temp_string_2;

  if (s[0]=='&' && s[length-1]==';') {
    // Output only valid escape codes for [0-9][a-f][A-F], ignore anything else.
    // &0x30; &0x31; ...? &48; &49; ...?
  }
}

/****************************************************************
 * This delimiter marks the beginning of exported "non-static"
 * functions.
 ****************************************************************/

/* Print any tool specific help */
void SHF_print_help(void) {
  /* Implement a generic charset translator */
  printf("         -i=shf           input SHF\n");
  printf("         -o=shf           output SHF\n");
}

/************************************************************************
 * Functions for writing SHF files
 ************************************************************************/

void SHF_write_header(char *name, u_int64_t blocks)
{
  OutString ("<dump name=\"%s\" blocks=\"%llx\">\n", name, blocks);
}

void SHF_write_block(shf_block_t *block)
{
  OutString("<block name=\"%s\" ", block->name);
  OutString("address=\"%016llx\" ", block->address);
  OutString("word_size=\"%llx\" ", block->word_length);
  OutString("length=\"%llx\" ", block->length);
  OutString("checksum=\"");
  dumpHexArray(block->checksum, 20, false);
  OutString("\">\n");
  OutString("<data>");
  // The bytes
  dumpHexArray(block->data, (block->length*block->word_length), false);  
  OutString("</data>\n");
  OutString("</block>\n");
}

void SHF_write_footer()
{
  OutString("</dump>\n");
}

/************************************************************************
 * Main function for reading SHF files
 ************************************************************************/

void SHF_process_file(FILE *infile)
{
  XML_Parser parser;
  bool done;
  int len;
  char Buff[BUFFSIZE];

  parser = XML_ParserCreate(NULL);
    
  if (! parser) {
    fprintf(stderr, "Could not create an XML parser instance\n");
    exit(1);
  }

  // Create an XML parser
  XML_SetElementHandler(parser, startElementHandler, endElementHandler);
  XML_SetCharacterDataHandler(parser, textHandler);
  XML_SetDefaultHandler(parser, defaultHandler);

  if (fseek(infile, 0, SEEK_SET)) {
    OutError("Could not read from input file!\n");
    exit(-1);
  }
  done = false;
  while (!done) {
    len = fread(Buff, 1, BUFFSIZE, infile);
    if (ferror(infile)) {
      OutError("Could not read from input file!\n");
      exit(-1);
    }
    done = feof(infile);
    
    if (! XML_Parse(parser, Buff, len, done)) {
      OutError("Parse error at line %d:\n%s\n",
	       XML_GetCurrentLineNumber(parser),
	       XML_ErrorString(XML_GetErrorCode(parser)));
      XML_ParserFree(parser);
      exit(-1);
    }
  }
  XML_ParserFree(parser);
}

/* End exported functions */
