/* cmdline.c
   Program for general command line interface
   Requires Expat (commonly packaged as xmltok-dev)
   Copyright (C) 2003 Linus Walleij

   This file is part of the shftool package.

   Shftool is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   You should have received a copy of the GNU General Public License
   along with Shftool; see the file COPYING.  If not, write to
   the Free Software Foundation, 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA. 

*/

#include "common.h"

/* Extra includes used by this file only */
#include <ctype.h>
#include <stdarg.h>
#include <dirent.h>
#include <sys/stat.h>

// Inputs and outputs
static int input_type = SHF_INPUT;
static int output_type = HEX_OUTPUT;

// input/output file
static FILE* InFile;
static FILE* OutFile;

void OutString(char *fmt, ...)
{
  va_list ap;
 
  va_start(ap, fmt);
  vfprintf(OutFile, fmt, ap);
  va_end(ap);
}

void OutError(char *fmt, ...)
{
  va_list ap;
 
  va_start(ap, fmt);
  vfprintf(stderr, fmt, ap);
  va_end(ap);
}

void OutChar(char c, bool err)
{
  if (!err) {
    fputc(c, OutFile);
  } else {
    fputc(c, stderr);
  }
}

/************************************************************************
 * Switching station
 * selects writing function depending on selected output filter
 ************************************************************************/

void write_header(char *name, u_int64_t blocks) {
  switch (output_type) {
  case SHF_OUTPUT:
    SHF_write_header(name, blocks);
    break;
  case HEX_OUTPUT:
    HEX_write_header(name, blocks);
    break;
  }
}

void write_block(shf_block_t *block) {
  switch (output_type) {
  case SHF_OUTPUT:
    SHF_write_block(block);
    break;
  case HEX_OUTPUT:
    HEX_write_block(block);
    break;
  }
}

void write_footer() {
  switch (output_type) {
  case SHF_OUTPUT:
    SHF_write_footer();
    break;
  case HEX_OUTPUT:
    HEX_write_footer();
    break;
  }
}


/************************************************************************
 * Main function call to translators
 ************************************************************************/

int
main(const int argc, const char **argv) {
  int arg;
  // Signifies if the program should write to stdout instead of a file
  bool Stdout = false;
  // Dump to output
  bool DumpMode = false;
  char cInFile[STR_MAXLEN] = "";
  char cOutFile[STR_MAXLEN] = "";
  static char cProgramName[]= PACKAGE "-" VERSION;

  if (argc<2) {
    printf("Usage: %s [-options] <file.shf>\n", argv[0]);
    printf("Type %s -help for all options.\n", argv[0]);
    exit(0);
  }

  for (arg=1; arg<argc; arg++) {
    if (*argv[arg] == '-') {
      // -? or -help: show the command line help page
      if (!strcmp(argv[arg],"-?") || !strcmp(argv[arg],"-help") || !strcmp(argv[arg],"--help")) {
	printf("Usage: %s [-options] <file[.shf]>\n\n", argv[0]);
	printf("Options: -                write to stdout\n");
	// Print any tool-specific help text
	SHF_print_help();
	HEX_print_help();
	exit(0);
	// non-interactive
      }
      else if (!strcmp(argv[arg],"-")) {
	Stdout = true;
      }
      // Check inputs and outputs - do this in an elegant way instead...
      else if (!strcmp(argv[arg],"-i=shf")) {
	input_type=SHF_INPUT;
      }
      else if (!strcmp(argv[arg],"-o=shf")) {
	output_type=SHF_OUTPUT;
      }
      else if (!strcmp(argv[arg],"-o=hex")) {
	output_type=HEX_OUTPUT;
      }
    }
  }

  // See that the source is OK
  strcpy(cInFile, argv[argc-1]);
  if (strcmp(cInFile+(strlen(cInFile)-4), ".shf"))
    strcpy(cInFile+strlen(cInFile), ".shf");
  
  if(!Stdout) {
    strcpy(cOutFile, cInFile);
    strcpy(cOutFile+strlen(cOutFile)-3,argv[0]+strlen(argv[0])-3);
    OutError("Input file: %s\n", cInFile);
    OutError("Output file: %s\n", cOutFile);
  }

  // Scan for the file or just assume it is here and open it 
  InFile = fopen(cInFile, "r");
  if (InFile == NULL) {
    OutError("Could not open input file <%s!>\n", cInFile);
    exit(-1);
  }

  if(!Stdout) {
    OutFile = fopen(cOutFile, "w");
    if (OutFile == NULL) {
      OutError("Could not create output file <%s>!\n", cOutFile);
      exit(-1);
    }
  } else {
    OutFile = stdout;
  }

  // Process the input file with the corresponding filter
  switch(input_type) {
  case SHF_INPUT:
    SHF_process_file(InFile);
    break;
  default:
    OutError("Unknown input type.\n");
    break;
  }

  fclose(InFile);
  if (!Stdout)
    fclose(OutFile);
  exit(0);
}
